/** Express router providing Contacto related routes
 * @module routers/Contacto
 * @requires express
 * @requires config - app config
 * @requires utils - app utils functions
 * @requires express-validator - form validation module
 * @requires models- app model module
 */


 /**
 * express module
 * @const
 */
const express = require('express');


/**
 * Express router to mount user page functions.
 * @type {object}
 * @const
 */
const router = express.Router();


/**
 * App config module
 * @const
 */
const config = require('../config.js');


/**
 * App utils functions module
 * @const
 */
const utils = require('../helpers/utils.js');


/**
 * Form input validation module
 * @const
 */
const { body, validationResult } = require('express-validator');


/**
 * Contacto models
 * @const
 */
const models = require('../models/index.js');
const Contacto = models.Contacto;


const sequelize = models.sequelize; // sequelize functions and operations
const Op = models.Op; // sequelize query operators


Contacto.belongsTo(models.Grupos, {foreignKey: 'id_grupo', as: 'grupos' });


/**
 * Route to list contacto records
 * @route {GET} /contacto/index/{fieldname}/{fieldvalue}
 * @param {array} path - Array of express paths
 * @param {callback} middleware - Express middleware.
 */
router.get(['/', '/index/:fieldname?/:fieldvalue?'], async (req, res) => {  
	try{
		let query = {};  // sequelize query object
		let where = {};  // sequelize where conditions
		let replacements = {};  // sequelize query params
		let fieldname = req.params.fieldname;
		let fieldvalue = req.params.fieldvalue;
		
		if (fieldname){
			where[Op.and] = [
				sequelize.literal(`(${fieldname} = :fieldvalue)`)
			];
			replacements.fieldvalue = fieldvalue;
		}
		let joinTables = []; // hold list of join tables
		joinTables.push({
			model: models.Grupos,
			required: false,
			as: 'grupos',
			attributes: [], //already set on the query attributes using sequelize literal
		})
		query['include'] = joinTables;
		let search = req.query.search;
		if(search){
			let searchFields = Contacto.searchFields();
			where[Op.or] = searchFields;
			replacements.search = `%${search}%`;
		}
		where['id_user'] = req.user.id; //filter only current records
		
		
		query.raw = true;
		query.where = where;
		query.replacements = replacements;
		query.order = Contacto.getOrderBy(req);
		query.attributes = Contacto.listFields();
		let page = parseInt(req.query.page) || 1;
		let limit = parseInt(req.query.limit) || 20;
		let result = await Contacto.paginate(query, page, limit);
		return res.ok(result);
	}
	catch(err) {
		return res.serverError(err);
	}
});


/**
 * Route to view Contacto record
 * @route {GET} /contacto/view/{recid}
 * @param {array} path - Array of express paths
 * @param {callback} middleware - Express middleware.
 */
router.get(['/view/:recid'], async (req, res) => {
	try{
		let recid = req.params.recid || null;
		let query = {}
		let where = {}
		let joinTables = []; // hold list of join tables
		joinTables.push({
			model: models.Grupos,
			required: false,
			as: 'grupos',
			attributes: [], //already set on the query attributes using sequelize literal
		})
		query['include'] = joinTables;
		where['id_user'] = req.user.id; //filter only current records
		where[Op.and] = sequelize.literal('contacto.id = :recid');
		query.replacements = {
			recid
		}
		query.raw = true;
		query.where = where;
		query.attributes = Contacto.viewFields();
		let record = await Contacto.findOne(query);
		if(!record){
			return res.notFound();
		}
		return res.ok(record);
	}
	catch(err){
		return res.serverError(err);
	}
});


/**
 * Route to insert Contacto record
 * @route {POST} /contacto/add
 * @param {string} path - Express path
 * @param {callback} middleware - Express middleware.
 */
router.post('/add/' , 
	[
		body('nome').optional(),
		body('telefone').optional(),
		body('id_grupo').optional(),
	]
, async function (req, res) {
	try{
		let errors = validationResult(req); // get validation errors if any
		if (!errors.isEmpty()) {
			let errorMsg = utils.formatValidationError(errors.array());
			return res.badRequest(errorMsg);
		}
		let modeldata = req.body;
		modeldata['id_user'] = req.user.id;
		
		//save Contacto record
		let record = await Contacto.create(modeldata);
		//await record.reload(); //reload the record from database
		let recid =  record['id'];
		
		return res.ok(record);
	} catch(err){
		return res.serverError(err);
	}
});


/**
 * Route to get  Contacto record for edit
 * @route {GET} /contacto/edit/{recid}
 * @param {string} path - Express path
 * @param {callback} middleware - Express middleware.
 */
router.get('/edit/:recid', async (req, res) => {
	try{
		let recid = req.params.recid;
		let query = {};
		let where = {};
		where['id_user'] = req.user.id; //filter only current records
		where['id'] = recid;
		query.where = where;
		query.attributes = Contacto.editFields();
		let record = await Contacto.findOne(query);
		if(!record){
			return res.notFound();
		}
		return res.ok(record);
	}
	catch(err){
		return res.serverError(err);
	}
});


/**
 * Route to update  Contacto record
 * @route {POST} /contacto/edit/{recid}
 * @param {string} path - Express path
 * @param {callback} middleware - Express middleware.
 */
router.post('/edit/:recid' , 
	[
		body('nome').optional(),
		body('telefone').optional(),
		body('id_grupo').optional(),
	]
, async (req, res) => {
	try{
		let errors = validationResult(req); // get validation errors if any
		if (!errors.isEmpty()) {
			let errorMsg = utils.formatValidationError(errors.array());
			return res.badRequest(errorMsg);
		}
		let recid = req.params.recid;
		let modeldata = req.body;
		let query = {};
		let where = {};
		where['id_user'] = req.user.id; //filter only current records
		where['id'] = recid;
		query.where = where;
		query.attributes = Contacto.editFields();
		let record = await Contacto.findOne(query);
		if(!record){
			return res.notFound();
		}
		await Contacto.update(modeldata, {where: where});
		return res.ok(modeldata);
	}
	catch(err){
		return res.serverError(err);
	}
});


/**
 * Route to delete Contacto record by table primary key
 * Multi delete supported by recid separated by comma(,)
 * @route {GET} /contacto/delete/{recid}
 * @param {array} path - Array of express paths
 * @param {callback} middleware - Express middleware.
 */
router.get('/delete/:recid', async (req, res) => {
	try{
		let recid = req.params.recid || '';
		recid = recid.split(',');
		let query = {};
		let where = {};
		where['id_user'] = req.user.id; //filter only current records
		where['id'] = recid;
		query.where = where;
		let records = await Contacto.findAll(query);
		records.forEach(async (record) => { 
			await record.destroy();
		});
		return res.ok(recid);
	}
	catch(err){
		return res.serverError(err);
	}
});
module.exports = router;
